<?php
/**
 * Service业务逻辑类
 */
class Service {
	protected $name = '';
	protected $db = null;
	protected $table = null;
	protected $pk = 'id';
	protected $db_config = '';
	protected $error = '';
	protected $options = array ();
	
	/**
	 * 架构函数
	 * @access public
	 */
	public function __construct() {
	}
	/** 
	 * 取得类实例
	 * @static
	 * @access public
	 * @param string $name 模型名称
	 * @param mixed $connection 数据库连接信息
	 * @return Service 返回类实例
	 */
	public static function instance($name = '', $connection = '') {
		$args = func_get_args ();
		static $_instance = array ();
		$identify = Guid ( $args );
		if (! isset ( $_instance [$identify] )) {
			$class = $name . 'Service';
			if (class_exists ( $class )) {
				$service = new $class ();
			} else {
				$service = new Service ();
			}
			$service->init ( $name, $connection );
			$_instance [$identify] = $service;
		}
		return $_instance [$identify];
	}
	/**
	 * 初始化
	 * @access public
	 * @param string $name 模型名称
	 * @param mixed $connection 数据库连接信息
	 */
	public function init($name = '', $connection = '') {
		if (! empty ( $name )) {
			$this->name = $name;
		} elseif (empty ( $this->name )) {
			$this->name = $this->getServiceName ();
		}
		$this->name = strtolower ( $this->name );
		$this->db ( empty ( $this->db_config ) ? $connection : $this->db_config );
		$class = $this->name . 'Table';
		if (class_exists ( $class )) {
			$this->table = new $class ();
		} else {
			$this->table = new Table ();
			$this->buildTable ();
		}
		$this->_init ();
	}
	/**
	 * 连接数据库
	 * @access public
	 * @param mixed $config  数据库配置
	 * @return this
	 */
	public function db($config = '') {
		if (! empty ( $config ) && is_string ( $config ) && false === strpos ( $config, '/' )) {
			$config = Config::load ( $config );
		}
		$this->db = Db::instance ( $config );
		return $this;
	}
	/**
	 * 生成数据表类
	 * @access private
	 * @return void
	 */
	private function buildTable() {
		$fields = $this->db->getFields ( Config::get ( 'database.db_prefix' ) . $this->name );
		if (! $fields) {
			return false;
		}
		$config = array ('auto_increment' => false, 'tablename' => $this->name );
		foreach ( $fields as $key => $val ) {
			if ($val ['primary_key']) {
				$config ['primary_key'] = $key;
				if ($val ['auto_increment'])
					$config ['auto_increment'] = 'true';
			}
		}
		$this->table->config = $config;
		$this->table->fields = $fields;
	}
	
	/**
	 * 得到当前的名称
	 * @access public
	 * @return string
	 */
	public function getServiceName() {
		if (empty ( $this->name ))
			$this->name = substr ( get_class ( $this ), 0, - 7 );
		return $this->name;
	}
	/**
	 * 获取当前模型
	 */
	public function getTable() {
		return $this->table;
	}
	/**
	 * 设置当前模型
	 * @param Table $table 模型对象实例
	 * @return this
	 */
	public function setTable($table) {
		if ($table instanceof Table) {
			$this->table = $table;
		}
		return $this;
	}
	/**
	 * 设置模型对象值
	 * @access public
	 * @param mixed $data 数据
	 * @return this
	 */
	public function setData($data) {
		$this->table->setData ( $data );
		return $this;
	}
	
	/**
	 * 设置数据对象的值
	 * @access public
	 * @param string $name 名称
	 * @param mixed $value 值
	 * @return void
	 */
	public function __set($name, $value) {
		$this->table->data [$name] = $value;
	}
	
	/**
	 * 获取数据对象的值
	 * @access public
	 * @param string $name 名称
	 * @return mixed
	 */
	public function __get($name) {
		return isset ( $this->table->data [$name] ) ? $this->table->data [$name] : null;
	}
	
	/**
	 * 检测数据对象的值
	 * @access public
	 * @param string $name 名称
	 * @return boolean
	 */
	public function __isset($name) {
		return isset ( $this->table->data [$name] );
	}
	
	/**
	 * 销毁数据对象的值
	 * @access public
	 * @param string $name 名称
	 * @return void
	 */
	public function __unset($name) {
		unset ( $this->table->data [$name] );
	}
	
	/**
	 * 利用__call方法实现一些特殊的查询
	 * @access public
	 * @param string $method 方法名称
	 * @param array $args 调用参数
	 * @return mixed
	 */
	public function __call($method, $args) {
		if (in_array ( strtolower ( $method ), array ('count', 'sum', 'min', 'max', 'avg' ), true )) {
			$field = isset ( $args [0] ) ? $args [0] : '*';
			return $this->getField ( strtoupper ( $method ) . '(' . $field . ') AS bf_' . $method );
		} elseif (strtolower ( substr ( $method, 0, 10 ) ) == 'getfieldby') {
			$name = strtolower ( substr ( $method, 10 ) );
			$where [$name] = $args [0];
			return $this->where ( $where )->getField ( $args [1] );
		} elseif (strtolower ( substr ( $method, 0, 5 ) ) == 'getby') {
			$field = strtolower ( substr ( $method, 5 ) );
			$where [$field] = $args [0];
			return $this->where ( $where )->getOne ();
		} else {
			Halt ( __CLASS__ . '的' . $method . '方法不存在' );
			return;
		}
	}
	/**
	 * 初始化回调方法
	 */
	protected function _init() {
	}
	
	/**
	 * 对保存到数据库的数据进行处理
	 * @access protected
	 * @param mixed $data 要操作的数据
	 * @return boolean
	 */
	protected function _process($data) {
		if (! empty ( $this->table->fields )) {
			foreach ( $data as $key => $val ) {
				if (! in_array ( $key, array_keys ( $this->table->fields ), true )) {
					unset ( $data [$key] );
				} elseif (is_scalar ( $val )) {
					$this->_parseType ( $data, $key );
				} elseif (! $val ['isexp'] && is_array ( $val )) {
					$data [$key] = serialize ( $val );
				}
			}
		}
		return $data;
	}
	
	/**
	 * 清除缓存
	 */
	public function clearCache() {
		FileData ( $this->name, null );
		Cache::instance ( array ('path' => APP_FILE_QUERY_PATH . $this->name . '/' ) )->clear ();
	}
	/**
	 * 新增数据
	 * @access public
	 * @param mixed $data 数据
	 * @param array $options 表达式
	 * @param boolean $replace 是否replace
	 * @return mixed
	 */
	public function insert($data = '', $options = array(), $replace = false) {
		if (empty ( $data )) {
			if (! empty ( $this->table->data )) {
				$data = $this->table->data;
				$this->table->data = array ();
			} else {
				$this->error = "无效数据";
				return false;
			}
		}
		$options = $this->_parseOptions ( $options );
		$data = $this->_process ( $data );
		$result = $this->db->insert ( $data, $options, $replace );
		if (false !== $result) {
			$this->clearCache ();
			$insertId = $this->getInsertId ();
			if ($insertId) {
				$data [$this->getPk ()] = $insertId;
				return $insertId;
			}
		}
		return $result;
	}
	
	/**
	 * 更新数据
	 * @access public
	 * @param mixed $data 数据
	 * @param array $options 表达式
	 * @return boolean
	 */
	public function update($data = '', $options = array()) {
		if (empty ( $data )) {
			if (! empty ( $this->table->data )) {
				$data = $this->table->data;
				$this->table->data = array ();
			} else {
				$this->error = '无效数据';
				return false;
			}
		}
		$data = $this->_process ( $data );
		$options = $this->_parseOptions ( $options );
		$pk = $this->getPk ();
		if (! isset ( $options ['where'] )) {
			if (isset ( $data [$pk] )) {
				$where [$pk] = $data [$pk];
				$options ['where'] = $where;
				unset ( $data [$pk] );
			} else {
				$this->error = '无效数据';
				return false;
			}
		}
		if (is_array ( $options ['where'] ) && isset ( $options ['where'] [$pk] )) {
			$pkValue = $options ['where'] [$pk];
		}
		$result = $this->db->update ( $data, $options );
		if (false !== $result) {
			$this->clearCache ();
			if (isset ( $pkValue ))
				$data [$pk] = $pkValue;
		}
		return $result;
	}
	
	/**
	 * 删除数据
	 * @access public
	 * @param mixed $options 表达式
	 * @return mixed
	 */
	public function delete($options = array()) {
		if (empty ( $options ) && empty ( $this->options ['where'] )) {
			if (! empty ( $this->table->data ) && isset ( $this->table->data [$this->getPk ()] ))
				return $this->delete ( $this->table->data [$this->getPk ()] );
			else
				return false;
		}
		$pk = $this->getPk ();
		if (is_numeric ( $options ) || is_string ( $options )) {
			if (strpos ( $options, ',' )) {
				$where [$pk] = array ('IN', $options );
			} else {
				$where [$pk] = $options;
			}
			$options = array ();
			$options ['where'] = $where;
		}
		$options = $this->_parseOptions ( $options );
		if (is_array ( $options ['where'] ) && isset ( $options ['where'] [$pk] )) {
			$pkValue = $options ['where'] [$pk];
		}
		$result = $this->db->delete ( $options );
		if (false !== $result) {
			$data = array ();
			if (isset ( $pkValue ))
				$data [$pk] = $pkValue;
		}
		$this->clearCache ();
		return $result;
	}
	
	/**
	 * 查询数据集
	 * @access public
	 * @param array $options 表达式参数
	 * @return mixed
	 */
	public function select($options = array()) {
		if (is_string ( $options ) || is_numeric ( $options )) {
			$pk = $this->getPk ();
			if (strpos ( $options, ',' )) {
				$where [$pk] = array ('IN', $options );
			} else {
				$where [$pk] = $options;
			}
			$options = array ();
			$options ['where'] = $where;
		} elseif (false === $options) {
			$options = array ();
			$options = $this->_parseOptions ( $options );
			return '( ' . $this->db->buildSelectSql ( $options ) . ' )';
		}
		$options = $this->_parseOptions ( $options );
		$resultSet = $this->db->select ( $options );
		if (false === $resultSet) {
			return false;
		}
		if (empty ( $resultSet )) {
			return null;
		}
		return $resultSet;
	}
	
	/**
	 * 生成查询SQL
	 * @access public
	 * @param array $options 表达式参数
	 * @return string
	 */
	public function buildSql($options = array(), $type = 'select') {
		$options = $this->_parseOptions ( $options );
		return '( ' . $this->db->buildSql ( $options, $type ) . ' )';
	}
	
	/**
	 * 分析表达式
	 * @access protected
	 * @param array $options 表达式参数
	 * @return array
	 */
	protected function _parseOptions($options = array()) {
		if (is_array ( $options ))
			$options = array_merge ( $this->options, $options );
		if (! isset ( $options ['table'] )) {
			$options ['table'] = $this->table->getTableName ();
		}
		$this->options = array ();
		return $options;
	}
	/**
	 * 数据类型检测
	 * @access protected
	 * @param mixed $data 数据
	 * @param string $key 字段名
	 * @return void
	 */
	protected function _parseType(&$data, $key) {
		$fieldType = strtolower ( $this->table->fields [$key] ['type'] );
		if (false === strpos ( $fieldType, 'bigint' ) && false !== strpos ( $fieldType, 'int' )) {
			$data [$key] = intval ( $data [$key] );
		} elseif (false !== strpos ( $fieldType, 'float' ) || false !== strpos ( $fieldType, 'double' )) {
			$data [$key] = floatval ( $data [$key] );
		} elseif (false !== strpos ( $fieldType, 'bool' )) {
			$data [$key] = ( bool ) $data [$key];
		}
	}
	/**
	 * 获取所有数据
	 */
	public function getAll($isfromDb = false) {
		$result = FileData ( $this->name );
		if (! $result || $isfromDb == true) {
			if (isset ( $this->table->fields ['sort'] )) {
				$this->order ( 'sort asc' );
			}
			$items = $this->select ();
			$result = array ();
			if ($items) {
				foreach ( $items as $item ) {
					$result [$item ['id']] = $item;
				}
			}
			FileData ( $this->name, $result );
		}
		return $result;
	}
	
	/**
	 * 查询数据  只获取一条数据
	 * @access public
	 * @param mixed $options 表达式参数
	 * @return mixed
	 */
	public function getOne($options = array()) {
		if (is_numeric ( $options ) || is_string ( $options )) {
			$where [$this->getPk ()] = $options;
			$options = array ();
			$options ['where'] = $where;
		}
		$options ['limit'] = 1;
		$options = $this->_parseOptions ( $options );
		$resultSet = $this->db->select ( $options );
		if (false === $resultSet) {
			return false;
		}
		if (empty ( $resultSet )) {
			return null;
		}
		$this->table->data = $resultSet [0];
		return $this->table->data;
	}
	
	/**
	 * 设置记录的某个字段值
	 * 支持使用数据库字段和方法
	 * @access public
	 * @param string|array $field  字段名
	 * @param string $value  字段值
	 * @return boolean
	 */
	public function setField($field, $value = '') {
		if (is_array ( $field )) {
			$data = $field;
		} else {
			$data [$field] = $value;
		}
		return $this->update ( $data );
	}
	
	/**
	 * 字段值递增
	 * @access public
	 * @param string $field  字段名
	 * @param integer $step  增长值
	 * @return boolean
	 */
	public function setIncrement($field, $step = 1) {
		return $this->setField ( $field, array ('isexp' => true, 'value' => $field . '+' . $step ) );
	}
	
	/**
	 * 字段值递减
	 * @access public
	 * @param string $field  字段名
	 * @param integer $step  减少值
	 * @return boolean
	 */
	public function setDecrement($field, $step = 1) {
		return $this->setField ( $field, array ('isexp' => true, 'value' => $field . '-' . $step ) );
	}
	
	/**
	 * 获取一条记录的某个字段值
	 * @access public
	 * @param string $field  字段名
	 * @param string $spea  字段数据间隔符号 NULL返回数组
	 * @return mixed
	 */
	public function getField($field, $sepa = null) {
		$options ['field'] = $field;
		$options = $this->_parseOptions ( $options );
		$field = trim ( $field );
		if (strpos ( $field, ',' )) {
			if (! isset ( $options ['limit'] )) {
				$options ['limit'] = is_numeric ( $sepa ) ? $sepa : '';
			}
			$resultSet = $this->db->select ( $options );
			if (! empty ( $resultSet )) {
				$_field = explode ( ',', $field );
				$field = array_keys ( $resultSet [0] );
				$key = array_shift ( $field );
				$key2 = array_shift ( $field );
				$cols = array ();
				$count = count ( $_field );
				foreach ( $resultSet as $result ) {
					$name = $result [$key];
					if (2 == $count) {
						$cols [$name] = $result [$key2];
					} else {
						$cols [$name] = is_string ( $sepa ) ? implode ( $sepa, $result ) : $result;
					}
				}
				return $cols;
			}
		} else {
			if (true !== $sepa) {
				$options ['limit'] = is_numeric ( $sepa ) ? $sepa : 1;
			}
			$result = $this->db->select ( $options );
			if (! empty ( $result )) {
				if (true !== $sepa && 1 == $options ['limit'])
					return reset ( $result [0] );
				foreach ( $result as $val ) {
					$array [] = $val [$field];
				}
				return $array;
			}
		}
		return null;
	}
	
	/**
	 * 使用正则验证数据
	 * @access public
	 * @param string $value  要验证的数据
	 * @param string $rule 验证规则
	 * @return boolean
	 */
	public function regex($value, $rule) {
		$validate = array ('require' => '/.+/', 'email' => '/^\w+([-+.]\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*$/', 'url' => '/^http(s?):\/\/(?:[A-za-z0-9-]+\.)+[A-za-z]{2,4}(?:[\/\?#][\/=\?%\-&~`@[\]\':+!\.#\w]*)?$/', 'currency' => '/^\d+(\.\d+)?$/', 'number' => '/^\d+$/', 'zip' => '/^\d{6}$/', 'integer' => '/^[-\+]?\d+$/', 'double' => '/^[-\+]?\d+(\.\d+)?$/', 'english' => '/^[A-Za-z]+$/' );
		if (isset ( $validate [strtolower ( $rule )] ))
			$rule = $validate [strtolower ( $rule )];
		return preg_match ( $rule, $value ) === 1;
	}
	
	/**
	 * SQL查询
	 * @access public
	 * @param string $sql  SQL指令
	 * @param mixed $parse  是否需要解析SQL
	 * @return mixed
	 */
	public function query($sql, $parse = false) {
		if (! is_bool ( $parse ) && ! is_array ( $parse )) {
			$parse = func_get_args ();
			array_shift ( $parse );
		}
		$sql = $this->parseSql ( $sql, $parse );
		return $this->db->query ( $sql );
	}
	
	/**
	 * 执行SQL语句
	 * @access public
	 * @param string $sql  SQL指令
	 * @param mixed $parse  是否需要解析SQL
	 * @return false | integer
	 */
	public function execute($sql, $parse = false) {
		if (! is_bool ( $parse ) && ! is_array ( $parse )) {
			$parse = func_get_args ();
			array_shift ( $parse );
		}
		$sql = $this->parseSql ( $sql, $parse );
		return $this->db->execute ( $sql );
	}
	
	/**
	 * 解析SQL语句
	 * @access public
	 * @param string $sql  SQL指令
	 * @param boolean $parse  是否需要解析SQL
	 * @return string
	 */
	protected function parseSql($sql, $parse) {
		if (true === $parse) {
			$options = $this->_parseOptions ();
			$sql = $this->db->buildSql ( $options, $sql );
		} elseif (is_array ( $parse )) {
			$parse = array_map ( array ($this->db, 'escapeString' ), $parse );
			$sql = vsprintf ( $sql, $parse );
		} else {
			$sql = strtr ( $sql, array ('[TABLE]' => $this->table->getTableName (), '[PREFIX]' => Config::get ( 'database.db_prefix' ) ) );
		}
		return $sql;
	}
	
	/**
	 * 启动事务
	 * @access public
	 * @return void
	 */
	public function beginTransaction() {
		$this->commit ();
		$this->db->beginTransaction ();
		return;
	}
	
	/**
	 * 提交事务
	 * @access public
	 * @return boolean
	 */
	public function commit() {
		return $this->db->commit ();
	}
	
	/**
	 * 事务回滚
	 * @access public
	 * @return boolean
	 */
	public function rollback() {
		return $this->db->rollback ();
	}
	
	/**
	 * 返回数据库的错误信息
	 * @access public
	 * @return string
	 */
	public function getDbError() {
		return $this->db->getError ();
	}
	
	/**
	 * 返回最后插入的ID
	 * @access public
	 * @return string
	 */
	public function getInsertId() {
		return $this->db->getInsertId ();
	}
	
	/**
	 * 返回最后执行的sql语句
	 * @access public
	 * @return string
	 */
	public function getSql() {
		return $this->db->getSql ();
	}
	
	/**
	 * 获取主键名称
	 * @access public
	 * @return string
	 */
	public function getPk() {
		return isset ( $this->table->config ['primary_key'] ) ? $this->table->config ['primary_key'] : $this->pk;
	}
	
	/**
	 * 获取数据表字段信息
	 * @access public
	 * @return array
	 */
	public function getDbFields() {
		if (isset ( $this->options ['table'] )) {
			$fields = $this->db->getFields ( $this->options ['table'] );
			return $fields ? array_keys ( $fields ) : false;
		}
		return array_keys ( $this->table->fields );
	}
	
	/**
	 * 查询缓存
	 * @access public
	 * @param mixed $config 缓存配置
	 * @return this
	 */
	public function cache($config = array()) {
		$this->options ['cache'] = array_merge ( $config, array ('path' => APP_FILE_QUERY_PATH . $this->name . '/' ) );
		return $this;
	}
	
	/**
	 * 指定查询字段 支持字段排除
	 * @access public
	 * @param mixed $field
	 * @param boolean $except 是否排除
	 * @return this
	 */
	public function field($field, $except = false) {
		if (true === $field) {
			$fields = $this->getDbFields ();
			$field = $fields ? $fields : '*';
		} elseif ($except) {
			if (is_string ( $field )) {
				$field = explode ( ',', $field );
			}
			$fields = $this->getDbFields ();
			$field = $fields ? array_diff ( $fields, $field ) : $field;
		}
		$this->options ['field'] = $field;
		return $this;
	}
	
	/**
	 * 指定查询条件
	 * @access public
	 * @param mixed $where 条件表达式
	 * @param mixed $parse 预处理参数
	 * @return this
	 */
	public function where($where, $parse = null) {
		if (empty ( $where ))
			return $this;
		if (! is_null ( $parse ) && is_string ( $where )) {
			if (! is_array ( $parse )) {
				$parse = func_get_args ();
				array_shift ( $parse );
			}
			$parse = array_map ( array ($this->db, 'escapeString' ), $parse );
			$where = vsprintf ( $where, $parse );
		}
		if (is_string ( $where ) && ! empty ( $where )) {
			$where = array (array ('type' => 'and', 'string' => $where ) );
		}
		if (isset ( $this->options ['where'] )) {
			$this->options ['where'] = array_merge ( $this->options ['where'], $where );
		} else {
			$this->options ['where'] = $where;
		}
		return $this;
	}
	/**
	 * 指定查询条件 AND 
	 * @access public
	 * @param mixed $where 条件表达式
	 * @param mixed $parse 预处理参数
	 * @return this
	 */
	public function whereAnd($field, $op, $value) {
		$where = array ('type' => 'and', 'field' => $field, 'op' => $op, 'value' => $value );
		if (isset ( $this->options ['where'] )) {
			$this->options ['where'] = array_merge ( $this->options ['where'], $where );
		} else {
			$this->options ['where'] = $where;
		}
		return $this;
	}
	/**
	 * 指定查询条件 Or 
	 * @access public
	 * @param mixed $where 条件表达式
	 * @param mixed $parse 预处理参数
	 * @return this
	 */
	public function whereOr($field, $op, $value) {
		$where = array ('type' => 'or', 'field' => $field, 'op' => $op, 'value' => $value );
		if (isset ( $this->options ['where'] )) {
			$this->options ['where'] = array_merge ( $this->options ['where'], $where );
		} else {
			$this->options ['where'] = $where;
		}
		return $this;
	}
	/**
	 * 指定查询数量
	 * @access public
	 * @param mixed $offset 起始位置
	 * @param mixed $length 查询数量
	 * @return this
	 */
	public function limit($offset, $length = null) {
		$this->options ['limit'] = is_null ( $length ) ? $offset : $offset . ',' . $length;
		return $this;
	}
	/**
	 * 指定SQL distinct
	 * @access public
	 * @param string $params 配置参数
	 * @return this
	 */
	public function distinct($params) {
		$this->options ['distinct'] = $params;
		return $this;
	}
	/**
	 * 指定SQL table
	 * @access public
	 * @param string $params 配置参数
	 * @return this
	 */
	public function table($params) {
		$this->options ['table'] = $params;
		return $this;
	}
	/**
	 * 指定SQL order
	 * @access public
	 * @param string $params 配置参数
	 * @return this
	 */
	public function order($params) {
		$this->options ['order'] = $params;
		return $this;
	}
	/**
	 * 指定SQL having
	 * @access public
	 * @param string $params 配置参数
	 * @return this
	 */
	public function having($params) {
		$this->options ['having'] = $params;
		return $this;
	}
	/**
	 * 指定SQL group
	 * @access public
	 * @param string $params 配置参数
	 * @return this
	 */
	public function group($params) {
		$this->options ['group'] = $params;
		return $this;
	}
	/**
	 * 指定SQL lock
	 * @access public
	 * @param string $params 配置参数
	 * @return this
	 */
	public function lock($params) {
		$this->options ['lock'] = $params;
		return $this;
	}
	/**
	 * 指定SQL 注释
	 * @access public
	 * @param string $comment 注释
	 * @return this
	 */
	public function comment($comment) {
		$this->options ['comment'] = $comment;
		return $this;
	}
	/**
	 * 指定SQL join部分
	 * @access public
	 * @param mixed $join
	 * @return this
	 */
	public function join($join) {
		if (! empty ( $join )) {
			$this->options ['join'] [] = $join;
		}
		return $this;
	}
	
	/**
	 * 指定SQL union部分
	 * @access public
	 * @param mixed $union
	 * @return this
	 */
	public function union($union) {
		if (empty ( $union ))
			return $this;
		$this->options ['union'] [] = $union;
		return $this;
	}
	
	/**
	 * 设置错误
	 * @access protected
	 * @param string $msg 错误提示
	 * @return bool
	 */
	protected function error($msg) {
		$this->error = $msg;
		return false;
	}
	/**
	 * 返回错误信息
	 * @access public
	 * @return string
	 */
	public function getError() {
		return $this->error;
	}
}